<?php
if ( ! defined( 'ABSPATH' ) ) die;

final class Zapomatic_Overview {

    public $db;
    protected $age = 3;
    protected $view = 10;
    protected $error;

    public function init_db() {

        if ( $this->db ) return;
        require_once( 'class-database.php' );
        $this->db = new Zapomatic_Database;

    }

    public function get_exec_products_price_movements( array $products_data ) {

        $drops = [];
        $rises = [];
        $statics = [];
            foreach ( $products_data as $product_id => $product_data ) {
                $new_price = (int) $product_data['new_price'];
                $last_price = (int) $product_data['last_price'];
                if ( empty( $last_price ) || empty( $new_price ) ) {
                    continue;
                }
                $diff = ( $new_price - $last_price );
                if ( $diff < 0 ) $drops[ $product_id ] = $diff;
                elseif ( $diff > 0 ) $rises[ $product_id ] = $diff;
                else $statics[ $product_id ] = $diff;
            }
            return array( 'drops' => $drops, 'rises' => $rises, 'statics' => $statics );
            
    }

    public function get_average_positions( int $age ) {

        $this->init_db();
        $execs = $this->db->get_execs_by_age( $age, 'completed' );
        $ap = [];
        foreach ( $execs as $exec ) {
            $exec_products_data = $this->db->get_row_value( $exec, 'products_data' );
            $single_exec_ap = [];
            foreach ( $exec_products_data as $product_data ) {
                $my_price = $product_data['new_price'];
                if ( empty( $my_price ) ) continue;
                $toptable = $product_data['toptable'];
                if ( empty( $toptable ) ) continue;

                $execs_ap = [];
                $pos = 1;
                foreach ( $toptable as $competitor ) {
                   if ( $competitor['price'] >= $my_price ) break;
                   $pos++; 
                }
                $single_exec_ap[] = $pos;
            }
            $ap[] = number_format( array_sum( $single_exec_ap ) / count( $single_exec_ap ), 2 );
        }
        if ( $age === 1 ) return $ap[0];
        return $ap;

    }

    public function get_price_trends( $age ) {

        $this->init_db();
        $execs = $this->db->get_execs_by_age( $age, 'completed' );
        $dates = [];
        $totals = [];
        $diffs = [];
        $values = [];
        foreach ( $execs as $exec ) {
            $dates[] = $this->db->get_row_value( $exec, 'start_time' );
            $exec_products_data = $this->db->get_row_value( $exec, 'products_data' );
            $totals[] = count( $exec_products_data );
            $exec_value = 0;
            $exec_diff = 0;
            foreach ( $exec_products_data as $exec_product_data ) {
                $new_price = (int) $exec_product_data['new_price'];
                $exec_value += $new_price;
                $last_price = (int) $exec_product_data['last_price'];
                if ( empty($last_price) || empty($new_price) ) {
                    continue;
                }
                $exec_diff += ( $new_price - $last_price );
            }
            $diffs[] = $exec_diff;
            $values[] = $exec_value;
        }
        return array(
            'dates' => $dates,
            'totals' => $totals,
            'diffs'  => $diffs,
            'values' => $values
        );

    }

    public function map_exec_products_statuses( array $products_data, $count = false ) {

        $e = [];
        $w = [];
        $s = [];
        foreach ( $products_data as $product_id => $product_data ) {
            $product_statuses = $product_data['statuses'];
            foreach ( $product_statuses as $status_type => $status_value ) {
                $product_id = $product_id;
                if ( ! $args = array( 'calc_model' => $product_statuses['calc_model'] ) ) $args = null;
                switch ( Zapomatic_Updater::status_type_decode( $status_value, $args ) ) {
                    case 'ERROR':
                        if ( ! in_array( $product_id, $e ) ) array_push( $e, $product_id );
                        break;
                    case 'WARNING':
                        if ( ! in_array( $product_id, $w ) ) array_push( $w, $product_id );
                        break;
                    case 'SUCCESS':
                        if ( ! in_array( $product_id, $s ) ) array_push( $s, $product_id );
                        break;
                }
            }
        }
        return array(
            'errors' => $count ? count( $e ) : $e,
            'warnings' => $count ? count( $w ) : $w,
            'successes' => $count ? count( $s ) : $s
        );

    }

    protected function timestamp() {

        $tz = 'Israel';
        $timestamp = time();
        $dt = new DateTime( 'now', new DateTimeZone( $tz ) );
        $dt->setTimestamp( $timestamp ); 
        return $dt->format( 'd.m.Y H:i:s' );

    }
    
    public function print_overview() {
        
        $tab = $this->menu_tools_callback_tabs( isset($_GET['tab']) ? $_GET['tab'] : null );
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php _e( 'סקירה - זאפומטיק', 'zapomatic' ); ?></h1>
            <nav class="nav-tab-wrapper woo-nav-tab-wrapper">
                <a href="?page=zapomatic" class="nav-tab <?php if($tab['key']==='products_reports'):?>nav-tab-active<?php endif; ?>">דו"ח מוצרים</a>
                <a href="?page=zapomatic&tab=exec_history" class="nav-tab <?php if($tab['key']==='exec_history'):?>nav-tab-active<?php endif; ?>">היסטוריית סריקות</a>
                <a href="?page=zapomatic&tab=positions_competitors" class="nav-tab <?php if($tab['key']==='positions_competitors'):?>nav-tab-active<?php endif; ?>">מיקומים ומתחרים</a>
                <a href="?page=zapomatic&tab=runtime" class="nav-tab <?php if($tab['key']==='runtime'):?>nav-tab-active<?php endif; ?>">זמני ריצה</a>
                <a href="?page=zapomatic&tab=products_trends" class="nav-tab <?php if($tab['key']==='products_trends'):?>nav-tab-active<?php endif; ?>">מגמות מחיר</a>
            </nav>
            <div class="woocommerce">
                <?php echo $this->menu_tools_callback_group( $tab['key'] ); ?>
            </div>
        </div>
        <?php
        return;

    }

    protected function menu_tools_callback_tabs( $tab = null ) {

        switch ( $tab ) {
            case 'positions_competitors':
                return array(
                    'key'   => $tab,
                    'title' => __( 'מיקומים ומתחרים', 'zapomatic' )
                );
            case 'exec_history':
                return array(
                    'key'   => $tab,
                    'title' => __( 'היסטוריית סריקות', 'zapomatic' )
                );
            case 'runtime':
                return array(
                    'key'   => $tab,
                    'title' => __( 'זמני ריצה', 'zapomatic' )
                );
            case 'products_trends':
                return array(
                    'key'   => $tab,
                    'title' => __('מגמות מחיר', 'zapomatic')
                );
            default:
                return array(
                    'key'   => 'products_reports',
                    'title' => __('דו"ח מוצרים', 'zapomatic')
                );
        }

    }

    public function compare_scans( $date1, $date2 ) {

        $this->init_db();
        $y = $this->db->get_completed_exec_by_date( $date1 );
        $z = $this->db->get_completed_exec_by_date( $date2 );
        return [$y, $z];

    }

    public function get_products_data_totals( $products_data, $column = null ) {

        $i = 0;
        switch ( $column ) {
            case 'is_first':
                foreach ( $products_data as $single ) {
                    if ( !empty( $single['is_first'] ) ) $i++;
                }
                return $i;
            case 'isnt_first':
                foreach ( $products_data as $single ) {
                    if ( empty( $single['is_first'] ) ) $i++;
                }
                return $i;
            case 'is_minimum':
                foreach ( $products_data as $single ) {
                    if ( !empty( $single['is_minimum'] ) ) $i++;
                }
                return $i;
            case 'isnt_minimum':
                foreach ( $products_data as $single ) {
                    if ( empty( $single['is_minimum'] ) ) $i++;
                }
                return $i;
            case 'is_assigned':
                foreach ( $products_data as $single ) {
                    if ( !empty( $single['is_assigned'] ) ) $i++;
                }
                return $i;
            case 'isnt_assigned':
                foreach ( $products_data as $single ) {
                    if ( empty( $single['is_assigned'] ) ) $i++;
                }
                return $i;
        }

    }

    public static function get_included_toptable( $product_id ) {

        function insertElement(&$array, $index, $element) {
            for ($i = count($array); $i > $index; $i--) {
                $array[$i] = $array[$i - 1];
            }
            $array[$index] = $element;
        }
        
        $product = @wc_get_product( $product_id );
        if ( !is_object( $product ) ) return false;
        $toptable = $product->get_meta( '_zapbott_pd_log_toptable' );
        
        if ( empty( $toptable ) ) return null;
        $my_price = @$product->get_price();
        
        if ( empty( $my_price ) ) return false;
        $pos = 1;
        foreach ( $toptable as $competitor ) {
           if ( $competitor['price'] >= $my_price ) break;
           $pos++; 
        }
        $pos = $pos-1;
        insertElement($toptable, $pos, array(
            'id'        => @get_option('zapbott_option_siteid'),
            'name'      => sprintf( '<b>%s</b>', get_bloginfo( 'name' ) ),
            'price'     => sprintf( '<b>%s</b>', $my_price ),
            'sale_type' => __( 'החנות שלך', 'zapomatic' )
        ) );
        return $toptable;

    }

    protected function menu_tools_callback_group( $tab = null ) {

        switch ( $tab ) {
            case 'positions_competitors':
                return $this->print_positions_competitors();
            case 'exec_history':
                return $this->print_exec_history();
            case 'runtime':
                return $this->print_runtime();
            case 'products_trends':
                return $this->print_products_trends();

            case 'products_reports':
                $this->init_db();
                echo '<form id="dates-filter" method="get">';
                printf(
                    '<input type="hidden" name="page" value="%s" />',
                    esc_attr( $_REQUEST['page'] )
                );
                $selected_date = isset( $_REQUEST['selected_date'] ) && !empty( $_REQUEST['selected_date'] ) ? $_REQUEST['selected_date'] : date('Y-m-d');
                printf(
                    '<label for="selected_date"><h3>%s:</h3></label>',
                    __( 'הצג סריקה אחרונה מתאריך', 'zapomatic' )
                );
                printf(
                    '<input type="date" name="selected_date" id="selected_date" value="%s" placeholder=""><br />',
                    $selected_date
                );
                submit_button( __( 'הצג', 'zapomatic' ), 'large', 'date_action', false );
                echo '</form>';
                if ( ! class_exists( 'WP_List_Table' ) ) require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
                $table = new Zapomatic_Overview_Table('last_exec');
                $table->selected_date = $selected_date;
                $table->prepare_items();
                $table->views();
                echo '<form id="posts-filter" method="get">';
                printf(
                    '<input type="hidden" name="page" value="%s" />',
                    esc_attr( $_REQUEST['page'] )
                );
                if ( isset($_REQUEST['selected_date'])) {
                    printf(
                        '<input type="hidden" name="selected_date" value="%s" />',
                        esc_attr( $_REQUEST['selected_date'] )
                    );
                }
                if ( isset($_REQUEST['subsubsub_filter'])) {
                    printf(
                        '<input type="hidden" name="subsubsub_filter" value="%s" />',
                        esc_attr( $_REQUEST['subsubsub_filter'] )
                    );
                }
               
                $table->search_box( __( 'חיפוש מוצרים', 'woocommerce' ), 'zapomatic-overview-search');
                $table->display();
                echo '</form>';
                return;
            default: 
                echo 'general';
                return;
        }
    }

    protected function print_products_trends() {

        $data = $this->process_localize( 'products_trends' );
        wp_localize_script( 'zapomatic_overview_pt', 'zapomatic_localized', $data );
        printf( '<h2>%s</h2>', __( 'מגמות מחיר', 'zapomatic' ) );
        $this->print_range_filter_form( array( 'page', 'tab' ) );
        ?>
        <div class="zapomatic-canvas-wrap">
        <canvas id="pt-canvas"></canvas>
        </div>
        <?php
        return;
        
    }

    protected function print_range_filter_form( array $args ) {

        $selected = (int) @$_REQUEST['selected_range'];
        echo '<form id="range-filter" method="get">';
                foreach ( $args as $r ) {
                    printf(
                        '<input type="hidden" name="%s" value="%s" />',
                        $r,
                        esc_attr( $_REQUEST[ $r ] )
                    );
                }
                printf(
                    '<label for="selected_range">%s:</label>',
                    __( 'טווח סריקות', 'zapomatic' )
                );
                echo '<select name="selected_range" id="selected_range" >';
                printf(
                    '<option value="7" %s>%s</option>',
                    $selected === 7 ? 'selected' : null,
                    __( 'שבוע', 'zapomatic' )
                );
                printf(
                    '<option value="14" %s>%s</option>',
                    $selected === 14 ? 'selected' : null,
                    __( 'שבועיים', 'zapomatic' )
                );
                printf(
                    '<option value="30" %s>%s</option>',
                    $selected === 0 || $selected === 30 ? 'selected' : null,
                    __( 'חודש', 'zapomatic' )
                );
                printf(
                    '<option value="183" %s>%s</option>',
                    $selected === 183 ? 'selected' : null,
                    __( 'חצי שנה', 'zapomatic' )
                );
                printf(
                    '<option value="365" %s>%s</option>',
                    $selected === 365 ? 'selected' : null,
                    __( 'שנה', 'zapomatic' )
                );
                echo '</select>';
                submit_button( __( 'הצג', 'zapomatic' ), '', 'range_action', false );
                echo '</form>';

    }

    protected function print_positions_competitors() {

        $data = $this->process_localize( 'positions_competitors' );
        wp_localize_script( 'zapomatic_overview_pc', 'zapomatic_localized', $data );
        printf( '<h2>%s</h2>', __( 'מיקומים ומתחרים', 'zapomatic' ) );
        $this->print_range_filter_form( array( 'page', 'tab' ) );
        ?>
        <div class="zapomatic-canvas-wrap">
        <canvas id="p-canvas"></canvas>
        </div>
        <div class="zapomatic-canvas-wrap">
        <canvas id="c-canvas"></canvas>
        </div>
        <?php
        return;
        
    }

    protected function print_exec_history() {

        $this->init_db();
        $data = $this->db->get_all_execs();
        if ( !empty( $data ) ) {
            ?>
            <table class="widefat fixed" cellspacing="0">
                <tr>
                    <th class="manage-column" scope="col"><?php _e('מועד התחלה', 'zapomatic'); ?></th>
                    <th class="manage-column" scope="col"><?php _e('מועד סיום', 'zapomatic'); ?></th>
                    <th class="manage-column" scope="col"><?php _e('עצירות', 'zapomatic'); ?></th>
                    <th class="manage-column" scope="col"><?php _e('מוצרים שנסרקו', 'zapomatic'); ?></th>
                    <th class="manage-column" scope="col"><?php _e('שגיאות', 'zapomatic'); ?></th>
                    <th class="manage-column" scope="col"><?php _e('אזהרות', 'zapomatic'); ?></th>
                    <th class="manage-column" scope="col"><?php _e('סטטוס', 'zapomatic'); ?></th>
                </tr>
                <?php
                foreach ( $data as $exec ) {
                    $products_data = unserialize( $exec->products_data );
                    $statuses = $this->map_exec_products_statuses( $products_data ,true );
                    echo '<tr class="alternate">';
                    printf( '<td class="manage-column" scope="row">%s</td>', $this->get_formatted_datetime( $exec->start_time ) );
                    printf( '<td class="manage-column" scope="row">%s</td>', $this->get_formatted_datetime( $exec->end_time ) );
                    printf( '<td class="manage-column" scope="row">%s</td>', $exec->stops );
                    printf(
                        '<td class="manage-column" scope="row">%s%s %s %s</td>', 
                        isset( $exec->products_data ) ? count( $products_data ) : '0',
                        isset( $exec->products_data ) ? __( ' מתוך', 'zapomatic' ) : null,
                        isset( $exec->total ) ? $exec->total : null,
                        __( 'מוצרים', 'zapomatic' ),
                    );
                    printf( '<td class="manage-column" scope="row">%s</td>', $statuses['errors'] );
                    printf( '<td class="manage-column" scope="row">%s</td>', $statuses['warnings'] );
                    printf( '<td class="manage-column" scope="row">%s</td>', $this->decode_exec_status( $exec->status ) );
                    echo '</tr>';
                }
                ?>
            </table>
            <?php
        }
        else printf( '<p>%s.</p>', __( 'לא נמצאו סריקות','zapomatic') );

    }

    protected function print_runtime() {

        $data = $this->process_localize( 'runtime' );
        wp_localize_script( 'zapomatic_overview_runtime', 'zapomatic_localized', $data );
        printf( '<h2>%s</h2>', __( 'זמני ריצה', 'zapomatic' ) );
        $this->print_range_filter_form( array( 'page', 'tab' ) );
        ?>
        <div class="zapomatic-canvas-wrap">
        <canvas id="rt-canvas"></canvas>
        </div>
        <?php
        return;

    }

    public function get_runtime_data( int $age ) {

        $this->init_db();
        $execs = $this->db->get_execs_by_age( $age, 'completed' );
        if ( ! $execs ) return null;
        $data['labels']['products'] = __( 'מוצרים שנסרקו', 'zapmatic' );
        $data['labels']['runtime'] = __( 'משך סריקה', 'zapmatic' );
        $data['labels']['stops'] = __( 'עצירות', 'zapmatic' );
        $x = [];
        foreach ( $execs as $exec ) {
            $exec_id = $exec['exec_id'];
            $start_time = $exec['start_time'];
            $end_time = $exec['end_time'];
            $runtime = $this->get_runtime( $start_time, $end_time );
            $data['runtime'][] = strtotime( $runtime );
            $data['runtime_str'][] = $runtime;
            $data['stops'][] = (int) $exec['stops'];
            $data['products'][] = $this->db->get_completed_product_ids( $exec_id, true );
            $x[] = $start_time;
            
        }
        $data['labels']['x'] = $this->convert_datetime_to_ddmm( $x );
        $data['labels']['tooltips'] = $this->get_formatted_datetime( $x );
        return $data;
        
    }

    protected function generate_chart_color() {

        static $index = 0;
        $color_palette = [
            [230, 25, 75],  // Red
            [60, 180, 75],  // Green
            [255, 225, 25], // Yellow
            [0, 130, 200],  // Blue
            [245, 130, 48], // Orange
            [145, 30, 180], // Purple
            [70, 240, 240], // Cyan
            [240, 50, 230], // Magenta
            [210, 245, 60], // Lime
            [250, 190, 190],// Pink
            [0, 128, 128],  // Teal
            [230, 190, 255],// Lavender
            [170, 110, 40], // Brown
            [255, 250, 200],// Beige
            [128, 0, 0],    // Maroon
            [170, 255, 195],// Mint
            [128, 128, 0],  // Olive
            [255, 215, 180],// Coral
            [0, 0, 128],    // Navy
            [128, 128, 128] // Grey
        ];
        $color = $color_palette[$index % count($color_palette)];
        $index = ($index + 1) % count($color_palette);
        return array(
            'borderColor'     => sprintf('rgb(%d, %d, %d)', $color[0], $color[1], $color[2]),
            'backgroundColor' => sprintf('rgba(%d, %d, %d, 0.5)', $color[0], $color[1], $color[2]),
        );

    }

    public function process_localize( $type ) {

        if ( isset( $_REQUEST['selected_range'] ) ) $selected_range = (int) $_REQUEST['selected_range'];
        else $selected_range = 30;
        $this->init_db();
        $arr = array();
        switch ( $type ) {
            case 'positions_competitors':
                $c_data = $this->get_top_competitors( 30, 10 );
                $c_labels = [];
                $c_counts = [];
                $border_colors = [];
                $bg_colors = [];
                foreach ($c_data as $competitor_name => $count) {
                    $c_labels[] = $competitor_name;
                    $c_counts[] = $count;
                    $colors = $this->generate_chart_color();
                    $border_colors[] = $colors['borderColor'];
                    $bg_colors[] = $colors['backgroundColor'];
                }
                $c_datasets = [
                    (object) [
                        'data'            => $c_counts,
                        'borderColor'     => $border_colors,
                        'backgroundColor' => $bg_colors,
                        'borderWidth'     => 1,
                    ]
                ];
                $arr['c']['data']['labels'] = $c_labels;
                $arr['c']['data']['datasets'] = $c_datasets; 
                $p_data = $this->get_top_positions( $selected_range, 10 );
                $arr['p']['data']['labels'] = $this->convert_datetime_to_ddmm( $p_data['dates'] );
                $arr['p']['data']['tooltips_labels'] = $this->get_formatted_datetime( $p_data['dates'] );
                $my_name = get_bloginfo( 'name' );
                $p_datasets = [];
                foreach ( $p_data['data'] as $store_name => $data ) {
                    $colors = $this->generate_chart_color();
                    $p_border_colors = $colors['borderColor'];
                    $p_bg_colors = $colors['backgroundColor'];
                    $p_datasets[] = (object) [
                        'label'            => $store_name,
                        'data'            => $data,
                        'borderColor'     => $p_border_colors,
                        'backgroundColor' => $p_bg_colors,
                        'borderWidth'     => 2,
                        'borderDash' => $store_name == $my_name ? [5, 5] : [0],
                        'pointStyle' => 'circle',
                        'pointRadius' => 3,
                        'pointHoverRadius' => 5,
                    ];
                }
                $arr['p']['data']['datasets'] = $p_datasets;
                return $arr;
            case 'products_trends':
                $data = $this->get_products_trends( $selected_range, 20 );
                $arr['data']['labels'] = $data['dates'];
                $arr['data']['tooltips_labels'] = $data['tooltips_dates'];
                $datasets = [];
                foreach ( $data['data'] as $product_name => $data ) {
                    $colors = $this->generate_chart_color();
                    $border_colors = $colors['borderColor'];
                    $bg_colors = $colors['backgroundColor'];
                    $datasets[] = (object) [
                        'label'            => $product_name,
                        'data'             => $data['prices'],
                        'url'        => $data['url'],
                        'borderColor'      => $border_colors,
                        'backgroundColor'  => $bg_colors,
                        'borderWidth'      => 2,
                        'pointStyle'       => 'circle',
                        'pointRadius'      => 3,
                        'pointHoverRadius' => 5,
                    ];
                }
                $arr['data']['datasets'] = $datasets;
                return $arr;
            case 'runtime':
                $arr = $this->get_runtime_data( $selected_range );
                return $arr;
        }
        return false;

    }

    public function get_top_positions( int $age, int $max = 10 ) {

        $this->init_db();
        $data = $this->db->get_execs_by_age( $age, 'completed' );
        $competitors = $this->get_top_competitors( $age, $max, true );
        $my_name = get_bloginfo( 'name' );
        $arr = [];
        foreach ( $data as $exec ) {
            $exec_products_data = $this->db->get_row_value( $exec, 'products_data' );
            $exec_date = $this->db->get_row_value( $exec, 'start_time' );
            $exec_toptables = [];
            $products_full_toptable = [];
            foreach ( $exec_products_data as $product_id => $product_data ) {
                if ( ! $my_price = $product_data['new_price'] ) {
                    $this->db->update_error_msg(
                        sprintf(
                            __( "Value new_price couldn't found inside product data ID %d [get_top_positions]", 'zapomatic' ), 
                            $product_id
                        )
                    );
                    continue;
                }
                if ( ! $toptable = $product_data['toptable'] ) continue;
                $my_defined = false;
                $pos = 1;
                foreach ( $toptable as $competitor ) {
                    if ( ! $my_defined && $competitor['price'] >= $my_price ) {
                        $my_defined = true;
                        $products_full_toptable[] = array(
                            'name'     => $my_name,
                            'position' => $pos,
                        );
                        $pos++; 
                    }
                    $products_full_toptable[] = array(
                            'position' => $pos,
                            'name'     => $competitor['name']
                    );
                    $pos++;
                }
            }
            $merged_toptable = [];
            foreach ( $products_full_toptable as $store ) {
                if ( in_array( $store['name'], $competitors ) || $my_name == $store['name'] ) {
                    $store_name = $store['name'];
                    $merged_toptable[ $store_name ][] = (int) $store['position'];
                }
            }
            $calced_toptable = [];
            foreach ( $merged_toptable as $top_store => $pos_value ) {
                $total = array_sum( $merged_toptable[ $top_store ] );
                $count = count( $merged_toptable[ $top_store ] );
                $average = $total / $count;
                $calced_toptable[ $top_store ] = number_format( $average, 2 );
            }
            $arr['data'][] = $calced_toptable;
            $arr['dates'][] = $exec_date;
        }
        $t = [];
        foreach ( $arr['data'] as $exec ) {
            foreach ( $exec as $store_name => $store_ap ) {
                $t[ $store_name ][] = $store_ap;
            }
            
        }
        $arr['data'] = $t;
        return $arr;

    }

    public function get_top_competitors( int $age, int $max = 10, bool $names_only = false ) {
        
        $this->init_db();
        $execs = $this->db->get_execs_products_data( $age );
        $execs_count = count( $execs );
        $updateAverage = function( $count ) use ( $execs_count ) {
            return round( $count / $execs_count );
        };
        $names = [];
        foreach ( $execs as $exec ) {
                foreach ( $exec as $product ) {
                    if ( isset($product['toptable']) ) {
                        foreach ($product['toptable'] as $toptable_item) {
                            $names[] = $toptable_item['name'];
                        }
                    }
                }
        }
        $name_counts = array_count_values($names);
        arsort($name_counts);
        $top_names = array_slice($name_counts, 0, $max, true);
        $top_names =  array_map($updateAverage,$top_names);
        if ( $names_only === true ) {
            $competitor_names = [];
            foreach ($top_names as $competitor_name => $count) {
                $competitor_names[] = $competitor_name; 
            }
            return $competitor_names;
        }
        return $top_names;

    }

    public function get_products_trends( int $age, int $max ) {

        $this->init_db();
        $execs = $this->db->get_execs_by_age( $age, 'completed' );
        $dates = [];
        $tooltips_dates = [];
        $count = count( $execs );
        if ( $count < 2 ) return;
        foreach ( $execs as $exec ) {
            $date = $exec['start_time'];
            $dates[] = implode('',$this->convert_datetime_to_ddmm( $date ));
            $tooltips_dates[] = $this->get_formatted_datetime( $date );
        }
        $execs = $this->db->get_execs_products_data( $age, 'completed' );
        $products = [];
        foreach ( $execs as $exec ) {
            foreach ( $exec as $product_id => $product_data ) {
                $new_price = (int) $product_data['new_price'];
                $name = html_entity_decode(get_the_title( $product_id ));

                    $products[ $name ]['prices'][] = $new_price;
                    $url = esc_url( add_query_arg(
                        'post',
                        $product_id,
                        get_admin_url() . 'post.php'
                    ) );
                    $products[ $name ]['url'] = sprintf( '%s&action=edit', $url );
            }
        }
        foreach ( $products as $product_id => $product_data ) {
            $last_price = (int) $product_data['prices'][0];
            $new_price = (int) end( $product_data['prices'] );
            $diff = $new_price - $last_price;
            $per = ( $diff / $last_price ) * 100;
            $products[ $product_id ]['diff'] = $diff;
            $products[ $product_id ]['per'] = number_format( $per, 2);
        }
        uasort($products, function($a, $b) {
            $aPer = is_numeric($a['per']) ? $a['per'] : -INF;
            $bPer = is_numeric($b['per']) ? $b['per'] : -INF;
            return $bPer <=> $aPer;
        });
        $products = array_slice( $products, 0, $max, true );
        return array(
            'dates' => $dates,
            'tooltips_dates' => $tooltips_dates,
            'data' => $products,
        );

    }

    public static function convert_datetime_to_ddmm( $arr ) {

        if ( !is_array( $arr ) ) $arr = array( $arr );
        $arr_map = function( $date ) {
            return date( 'd/m', strtotime( $date ) );
        };
        return array_map( $arr_map, $arr );

    }

    public static function convert_datetime_to_days_ago( $arr ) {

        if ( !is_array( $arr ) ) $arr = array( $arr );
        $currentDate = new DateTime();
        $currentDate->setTime(0, 0);
        $convertToDaysAgo = function($date) use ($currentDate) {
            $dateObj = new DateTime($date);
            $dateObj->setTime(0, 0);
            $interval = $currentDate->diff($dateObj);
            $daysAgo = (int) $interval->format('%r%a');
            if ( $daysAgo == 0 ) {
                return __( 'היום', 'zapomatic' );
            } elseif ( $daysAgo == -1 ) {
                return __( 'אתמול', 'zapomatic' );
            } elseif ( $daysAgo == -2 ) {
                return __( 'שלשום', 'zapomatic' );
            } else {
                return sprintf( __( 'לפני %s ימים', 'zapomatic' ), abs($daysAgo) );
            }
        };
        return array_map( $convertToDaysAgo, $arr );

    }

    public static function get_formatted_datetime( $datetime_str ) {

        $date_format = get_option('date_format');
        $time_format = get_option('time_format');
        $combined_format = $date_format . ' ' . $time_format;
        if ( is_array( $datetime_str ) ) {
            $arr = [];
            foreach ( $datetime_str as $single ) {
                $arr[] = date_i18n( $combined_format, strtotime( $single ) );
            }
            return $arr;
        }
        $timestamp = strtotime($datetime_str);
        return date_i18n($combined_format, $timestamp);

    }

    public static function get_runtime( $start_time, $end_time ) {

        $start = new DateTime( $start_time );
        $end = new DateTime( $end_time );
        $interval = $start->diff( $end );
        return $interval->format( '%H:%I:%S' );
        
    }

    public static function decode_exec_status( $status ) {

        switch ( $status ) {
            case 'completed':
                return __( 'הושלמה', 'zapomatic' );
            case 'aborted':
                return __( 'בוטלה ידנית', 'zapomatic' );
            case 'canceled':
                return __( 'בוטלה', 'zapomatic' );
            case 'stopped':
                return __( 'נעצרה', 'zapomatic' );
            case 'in-progress':
                return __( 'בביצוע', 'zapomatic' );
            default:
                return __( 'לא ידוע', 'zapomatic' );
        }

    }

}

class Zapomatic_Overview_Table extends WP_List_Table {

    private $db;
    public $items;
    public $per_page = 100;
    protected $counters;
    protected $total;
    public $selected_date;

    public function __construct() {
        
        require_once('class-database.php');
        $this->db = new Zapomatic_Database;
        $this->counters['eilat'] = 0;
        parent::__construct( array(
            'singular' => __( 'Product', 'zapomatic' ),
            'plural'   => __( 'Products', 'zapomatic' ),
            'ajax'     => false,
        ) );

    }

    function get_data() {

            $data = $this->db->get_completed_exec_by_date( $this->selected_date );
            if ( ! $data ) return array();
            return $this->db->get_row_value( $data, 'products_data' );

    }

    function prepare_items() {

        $data = $this->get_data();
        $this->total = count( $data );
        $search_term = isset($_REQUEST['s']) ? trim($_REQUEST['s']) : '';
        $subsubsub_filter = isset( $_REQUEST['subsubsub_filter'] ) && $_REQUEST['subsubsub_filter'] != 'all' ? trim($_REQUEST['subsubsub_filter']) : '';
        $items = array();
        foreach ( $data as $product_id => $details ) {
            $row = array(
                'product_id'  => $product_id,
                'name'        => get_the_title( $product_id ),
                'last_price'  => $details['last_price'],
                'new_price'   => $details['new_price'],
                'price_diff'  => !empty( $details['new_price'] ) && !empty( $details['last_price'] ) ? ( $details['new_price'] - $details['last_price'] ) : null,
                'first_price' => get_post_meta( $product_id, '_zapbott_pd_log_firstprice', true ),
                'is_eilat' => !empty( $details['is_eilat'] ) ? true : false,
                'is_first' => !empty( $details['is_first'] ) ? true : false,
                'is_minimum'  => !empty( $details['is_minimum'] ) ? true : false,
                'is_assigned' => !empty( $details['is_assigned'] ) ? true : false,
                'statuses' => $details['statuses'],
                'model_url' => get_post_meta( $product_id, '_zapbott_pd_zap_modelurl', true ),
            );
            $items[] = $row;
            if ( !empty( $details['is_eilat'] ) ) $this->counters['eilat']++;
        }
        if ( !empty( $search_term ) ) {
            $search_results = [];
            foreach ( $items as $row ) {
                $search_match = strpos( $row['name'], $search_term );
                if ( is_numeric( $search_match ) ) {
                    $search_results[] = $row;
                }
            }
            $items = $search_results;
        }
        if ( isset( $_REQUEST['product_cat'] ) && !empty( $_REQUEST['product_cat'] ) ) {
            $product_cat = $_REQUEST['product_cat'];
            $filter_results = [];
            foreach ( $items as $row ) {
                if ( has_term( $_REQUEST['product_cat'], 'product_cat', $row['product_id'] ) ) $filter_results[] = $row;
            }
            $items = $filter_results;
        }
        if ( isset( $_REQUEST['is_minimum'] ) && !empty( $_REQUEST['is_minimum'] ) ) {
            $is_minimum = $_REQUEST['is_minimum'] == 'true' ? true : false;
            $filter_results = [];
            foreach ( $items as $row ) {
                if ($row['is_minimum'] == $is_minimum) $filter_results[] = $row;
            }
            $items = $filter_results;
        }
        if ( isset( $_REQUEST['is_first'] ) && !empty( $_REQUEST['is_first'] ) ) {
            $is_first = $_REQUEST['is_first'] == 'true' ? true : false;
            $filter_results = [];
            foreach ( $items as $row ) {
                if ($row['is_first'] == $is_first) $filter_results[] = $row;
            }
            $items = $filter_results;
        }
        if ( isset( $_REQUEST['is_assigned'] ) && !empty( $_REQUEST['is_assigned'] ) ) {
            $is_assigned = $_REQUEST['is_assigned'] == 'true' ? true : false;
            $filter_results = [];
            foreach ( $items as $row ) {
                if ($row['is_assigned'] == $is_assigned) $filter_results[] = $row;
            }
            $items = $filter_results;
        }
        if ( isset( $_REQUEST['is_alone'] ) && !empty( $_REQUEST['is_alone'] ) ) {
            $is_alone = $_REQUEST['is_alone'] == 'true' ? true : false;
            $filter_results = [];
            foreach ( $items as $row ) {
                if ( $is_alone ) {
                    if ( $row['statuses']['calc_model'] == 600 || $row['statuses']['calc_model'] == 610 ) $filter_results[] = $row;
                }
                else {
                    if ( isset( $row['statuses']['calc_model'] ) && $row['statuses']['calc_model'] != 600 && $row['statuses']['calc_model'] != 610 ) $filter_results[] = $row;
                }
            }
            $items = $filter_results;
        }
        if ( isset( $_REQUEST['price_trend'] ) && !empty( $_REQUEST['price_trend'] ) ) {
            $filter_results = [];
            foreach ( $items as $row ) {
                switch ( $_REQUEST['price_trend'] ) {
                    case 'up':
                        if ( $row['new_price'] > $row['last_price'] ) $filter_results[] = $row;
                        break;
                    case 'down':
                        if ( $row['new_price'] < $row['last_price'] ) $filter_results[] = $row;
                        break;
                    default:
                        if ( $row['new_price'] == $row['last_price'] ) $filter_results[] = $row;
                        break;
                }
            }
            $items = $filter_results;
        }
        if ( isset( $_REQUEST['status'] ) && !empty( $_REQUEST['status'] ) ) {
            $filter_results = [];
            foreach ( $items as $row ) {
                $statuses = $this->get_row_product_statuses( $row['statuses'] );
                if ( count( $statuses[ $_REQUEST['status'] ] ) > 0 ) $filter_results[] = $row;
            }
            $items = $filter_results;
        }
        if ( isset( $subsubsub_filter ) && !empty( $subsubsub_filter ) ) {
            $items = $this->handle_subsubsub_filter( $subsubsub_filter, $items );
        }
        $columns = $this->get_columns();
        $hidden = array();
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = array($columns, $hidden, $sortable);
        usort( $items, array( &$this, 'usort_reorder' ) );
        $current_page = $this->get_pagenum();
        $total_items = count($items);
        $this->set_pagination_args( array(
            'total_items' => $total_items,
            'per_page'    => $this->per_page,
            'total_pages' => ceil($total_items/$this->per_page)
        ) );
        $items = array_slice($items,(($current_page-1)*$this->per_page),$this->per_page); 
        $this->items = $items;
        
    }

    protected function handle_subsubsub_filter( $subsubsub, $items ) {

        $results = array();
        switch ( $subsubsub ) {
            case 'eilat':
                foreach ( $items as $row ) {
                    if ( !empty( $row['is_eilat'] ) ) $results[] = $row;
                }
                break;
        }
        return $results;

    }

    function usort_reorder( $a, $b ) {

        $orderby = ( ! empty( $_GET['orderby'] ) ) ? $_GET['orderby'] : 'name';
        $order = ( ! empty($_GET['order'] ) ) ? $_GET['order'] : 'asc';
        $result = strcmp( $a[$orderby], $b[$orderby] );
        return ( $order === 'asc' ) ? $result : -$result;
    }

    public function get_views() { 

        $status_links = array(
            "all"        => sprintf( '<a href="admin.php?page=zapomatic&selected_date=%s&subsubsub_filter=all" class="%s">%s <span class="count">(%d)</span></a>', isset( $_REQUEST['selected_date'] ) ? urlencode($_REQUEST['selected_date']) : null, !isset( $_REQUEST['subsubsub_filter'] ) || $_REQUEST['subsubsub_filter'] == 'all' ? 'current' : '', __( 'הכל', 'zapomatic' ), $this->total ),
            'eilat' => sprintf( '<a href="admin.php?page=zapomatic&selected_date=%s&subsubsub_filter=eilat" class="%s">%s <span class="count">(%d)</span></a>', isset( $_REQUEST['selected_date'] ) ? urlencode($_REQUEST['selected_date']) : null, isset( $_REQUEST['subsubsub_filter'] ) && $_REQUEST['subsubsub_filter'] == 'eilat' ? 'current' : '', __( 'אילת בלבד', 'zapomatic' ), $this->counters['eilat'] ),
        );
        return $status_links;
    }


    function get_columns() {

        $columns = array(
            'thumb'       => __( 'תמונה', 'woocommerce' ),
            'name'        => __( 'שם', 'woocommerce' ),
            'new_price'   => __( 'מחיר חדש', 'zapomatic' ),
            'last_price'  => __( 'מחיר קודם', 'zapomatic' ),
            'price_diff'  => __( 'שינוי', 'zapomatic' ),
            'statuses'  => __( 'סטטוסים', 'zapomatic' ),
            'is_minimum'  => __( 'הגיע למינימום', 'zapomatic' ),
            'is_assigned' => __( 'משוייך למודל', 'zapomatic' ),
            'is_first' => __( '#1 בזאפ', 'zapomatic' ),
            'first_price' => __( 'מחיר ראשון בזאפ', 'zapomatic' ),
            'model_url' => __( 'עמוד מודל', 'zapomatic' ),
            // 'toptable'     => __( 'Top Table Retailers', 'zapomatic' ),
        );
        return $columns;
    }

    // Render each column
    function column_default( $item, $column_name ) {
        switch ( $column_name ) {
            case 'thumb':
                return $this->get_product_thumbnail( $item['product_id'] );
            case 'name':
                $url = esc_url( add_query_arg(
                    'post',
                    $item['product_id'],
                    get_admin_url() . 'post.php'
                ) );
                return sprintf( '<strong><a class="row-title" href="%s&action=edit" target="_blank">%s</a><strong>', $url, $item[ $column_name ] );
            case 'price_diff':
                return !empty( $item[ $column_name ] ) ? sprintf('<span style="color:%s;"><b>%s</b></span>', $item[ $column_name ] >= 0 ? 'green' : 'red', wc_price( $item[ $column_name ] )) : null;
            case 'last_price':
            case 'new_price':
            case 'first_price':
                return !empty( $item[ $column_name ] ) ? wc_price( $item[ $column_name ] ) : null;
            case 'statuses':
                return $this->print_row_statuses( $item[ $column_name ] );
            case 'is_minimum':
            case 'is_assigned':
            case 'is_first':
                return $item[ $column_name ] === true ? '&#10004;' : '&#10006;';
            // case 'toptable':
            //     return $item[ $column_name ];
            case 'model_url':
                return sprintf('<a class="row-title" href="%s" target="_blank">&#8632;</a>', $item[ $column_name ] );
            default:
                return print_r( $item, true ); // Show the whole array for unexpected columns
        }
    }
    function get_bulk_actions() {

        $actions = array(
        'mail' => __( 'שלח סריקה למייל', 'pbfwoo' ),
        'delete' => __( 'מחק סריקה"', 'pbfwoo' ),
        );
        return $actions;

    }
    function get_sortable_columns() {

        return array(
            'name'  => array( 'name', true ),
            'last_price'  => array( 'last_price', false ),
            'new_price'   => array( 'new_price', false ),
            'price_diff'  => array( 'price_diff', false ),
            'is_minimum'  => array( 'is_minimum', false ),
            'is_assigned' => array( 'is_assigned', false ),
            'first_price' => array( 'first_price', false ),
            'is_first' => array( 'is_first', false ),
        );
    }

    private function get_product_thumbnail( $product_id ) {
        $product = wc_get_product( $product_id );
        if ( !$product ) return null;
        // Retrieve the image ID associated with the product
        $image_id = $product->get_image_id();
        // Get the image URL using the image ID and specify the image size ('full' in this case)
        return wp_get_attachment_image( $image_id, 'thumbnail', true );
    }

    public function get_row_product_statuses( array $product_statuses ) {

        $e = [];
        $w = [];
        $s = [];
        foreach ( $product_statuses as $status_type => $status_code ) {
            if ( ! $args = array( 'calc_model' => $product_statuses['calc_model'] ) ) $args = null;
            switch ( Zapomatic_Updater::status_type_decode( $status_code, $args ) ) {
                case 'ERROR':
                    array_push( $e, $status_code );
                    break;
                case 'WARNING':
                    array_push( $w, $status_code );
                    break;
                case 'SUCCESS':
                    array_push( $s, $status_code );
                    break;
            }
        }
        return array(
            'errors' => $e,
            'warnings' => $w,
            'successes' => $s,
        );

    }
    
    public function print_row_statuses( array $product_statuses ) {

        $e = [];
        $e_ol = 1;
        $w = [];
        $w_ol = 1;
        $s = [];
        $s_ol = 1;
        $result = '';
        foreach ( $product_statuses as $status_type => $status_code ) {
            if ( ! $args = array( 'calc_model' => $product_statuses['calc_model'] ) ) $args = null;
            switch ( Zapomatic_Updater::status_type_decode( $status_code, $args ) ) {
                case 'ERROR':
                    array_push( $e, sprintf( '%d. %s', $e_ol, Zapomatic_Updater::status_decode( $status_code, $args )['info'] ) );
                    $e_ol++;
                    break;
                case 'WARNING':
                    array_push( $w, sprintf( '%d. %s', $w_ol, Zapomatic_Updater::status_decode( $status_code, $args )['info'] ) );
                    $w_ol++;
                    break;
                case 'SUCCESS':
                    array_push( $s, sprintf( '%d. %s', $s_ol, Zapomatic_Updater::status_decode( $status_code, $args )['info'] ) );
                    $s_ol++;
                    break;
            }
        }
        if ( !empty( $s ) ) {
            $result .= sprintf(
                '<span title="%s" class="status success">%d</span>',
                implode( PHP_EOL, $s ),
                count( $s )
            );
        }
        if ( !empty( $w ) ) {
            $result .= sprintf(
                '<span title="%s" class="status warning">%d</span>',
                implode( PHP_EOL, $w ),
                count( $w )
            );
        }
        if ( !empty( $e ) ) {
            $result .= sprintf(
                '<span title="%s" class="status error">%d</span>',
                implode( PHP_EOL, $e ),
                count( $e )
            );
        }
        return $result;

    }

    // Set no items message
    function no_items() {
        _e( 'לא נמצאו מוצרים.', 'zapomatic' );
    }
    function extra_tablenav( $which ) {
        if ( $which == 'top' ) { // Only display the dropdown above the table
            // Wrap in the "alignleft actions" class, just like WooCommerce
            echo '<div class="alignleft actions">';
            // Add a custom dropdown (for example, to filter by stock status)
            $selected_minimum = isset( $_REQUEST['is_minimum'] ) ? $_REQUEST['is_minimum'] : '';
            $selected_assigned = isset( $_REQUEST['is_assigned'] ) ? $_REQUEST['is_assigned'] : '';
            $selected_alone = isset( $_REQUEST['is_alone'] ) ? $_REQUEST['is_alone'] : '';
            $selected_first = isset( $_REQUEST['is_first'] ) ? $_REQUEST['is_first'] : '';
            $selected_status = isset( $_REQUEST['status'] ) ? $_REQUEST['status'] : '';
            $selected_price_trend = isset( $_REQUEST['price_trend'] ) ? $_REQUEST['price_trend'] : '';
            $selected_category = isset( $_REQUEST['product_cat'] ) ? $_REQUEST['product_cat'] : '';
            echo '<select name="price_trend">';
            echo '<option value="">' . __( 'סנן לפי מגמת מחיר', 'zapomatic' ) . '</option>';
            echo '<option value="up"' . selected( $selected_price_trend, 'up', false ) . '>' . __( 'עליית מחיר', 'zapomatic' ) . '</option>';
            echo '<option value="down"' . selected( $selected_price_trend, 'down', false ) . '>' . __( 'ירידת מחיר', 'zapomatic' ) . '</option>';
            echo '<option value="static"' . selected( $selected_price_trend, 'static', false ) . '>' . __( 'ללא שינוי מחיר', 'zapomatic' ) . '</option>';
            echo '</select>';
            echo '<select name="status">';
            echo '<option value="">' . __( 'סנן לפי סטטוס', 'zapomatic' ) . '</option>';
            echo '<option value="errors"' . selected( $selected_status, 'errors', false ) . '>' . __( 'הצג שגיאות', 'zapomatic' ) . '</option>';
            echo '<option value="warnings"' . selected( $selected_status, 'warnings', false ) . '>' . __( 'הצג אזהרות', 'zapomatic' ) . '</option>';
            echo '<option value="successes"' . selected( $selected_status, 'successes', false ) . '>' . __( 'הצג הצלחות', 'zapomatic' ) . '</option>';
            echo '</select>';
            echo '<select name="is_minimum">';
            echo '<option value="">' . __( 'סנן לפי הגעה למינימום', 'zapomatic' ) . '</option>';
            echo '<option value="true"' . selected( $selected_minimum, 'true', false ) . '>' . __( 'הגיע למינימום', 'zapomatic' ) . '</option>';
            echo '<option value="false"' . selected( $selected_minimum, 'false', false ) . '>' . __( 'לא הגיע למינימום', 'zapomatic' ) . '</option>';
            echo '</select>';
            echo '<select name="is_first">';
            echo '<option value="">' . __( 'סנן לפי מקום ראשון', 'zapomatic' ) . '</option>';
            echo '<option value="true"' . selected( $selected_first, 'true', false ) . '>' . __( 'מקום ראשון', 'zapomatic' ) . '</option>';
            echo '<option value="false"' . selected( $selected_first, 'false', false ) . '>' . __( 'לא מקום ראשון', 'zapomatic' ) . '</option>';
            echo '</select>';
            echo '<select name="is_assigned">';
            echo '<option value="">' . __( 'סנן לפי שיוך לעמוד מודל', 'zapomatic' ) . '</option>';
            echo '<option value="true"' . selected( $selected_assigned, 'true', false ) . '>' . __( 'משוייך לעמוד מודל', 'zapomatic' ) . '</option>';
            echo '<option value="false"' . selected( $selected_assigned, 'false', false ) . '>' . __( 'לא משוייך לעמוד מודל', 'zapomatic' ) . '</option>';
            echo '</select>';
            echo '<select name="is_alone">';
            echo '<option value="">' . __( 'סנן לפי חנות בודדה', 'zapomatic' ) . '</option>';
            echo '<option value="true"' . selected( $selected_alone, 'true', false ) . '>' . __( 'בודדה בעמוד מודל', 'zapomatic' ) . '</option>';
            echo '<option value="false"' . selected( $selected_alone, 'false', false ) . '>' . __( 'לא בודדה בעמוד מודל', 'zapomatic' ) . '</option>';
            echo '</select>';
        
        wp_dropdown_categories( array(
            'show_option_all' => __( 'בחר קטגוריה', 'zapomatic' ),
            'taxonomy'        => 'product_cat',
            'name'            => 'product_cat',
            'orderby'         => 'name',
            'selected'        => $selected_category,
            'hierarchical'    => true,
            'show_count'      => true,
            'hide_empty'      => false,
            'value_field'     => 'term_id',
        ));
            // Add a submit button for the filter
            submit_button( __( 'סינון', 'zapomatic' ), '', 'filter_action', false );
            echo '</div>'; // End of the "alignleft actions" div
        }
    }
}
?>
