<?php
if ( ! defined( 'ABSPATH' ) ) die;

final class Zapomatic_Database {

    protected $wpdb;
    protected $table;
    protected $exec_id;

    public function __construct() {

        global $wpdb;
        $this->wpdb = $wpdb;
        $this->table = $this->wpdb->prefix . 'zapomatic_logs';
        if ( $zapbott_full_exec_id = $this->get_active_exec_id() ) $this->exec_id = $zapbott_full_exec_id;
        if ( $wpdb->get_var( "SHOW TABLES LIKE '$this->table'" ) !== $this->table ) $this->create_db_table(); 
        
    }

    public function __destruct() {

        if ( $this->wpdb->last_error ) error_log( print_r( $this->wpdb->last_error, true ) );

    }

    public function update_error_msg( string $str ) {
        
        $error_msg = sprintf( '[%s] Zapomatic Database Error: %s', date('d-M-Y H:i:s T'), $str );
        error_log( $error_msg );

    }

    private function limit_db_rows( $max_rows ) {

        $max_rows = round( $max_rows * 30.44 );
        $current_count = $this->wpdb->get_var( "SELECT COUNT(*) FROM $this->table" );
        if ( $current_count <= $max_rows ) return null;
        $rows = $current_count - $max_rows;
        $execs = $this->wpdb->get_results( "SELECT exec_id FROM $this->table ORDER BY start_time ASC LIMIT $rows" );
        foreach ( $execs as $exec ) {
            $this->wpdb->delete( $this->table, [ 'exec_id' => $exec->exec_id ], [ '%s' ] );
        }
        return true;

    }

    public function create_db_table() {
        
        $charset_collate = $this->wpdb->get_charset_collate();
        $sql = "CREATE TABLE $this->table (
          exec_id VARCHAR(255) NOT NULL,
          start_time datetime NOT NULL,
          end_time datetime,
          stops smallint,
          total smallint NOT NULL,
          status tinytext NOT NULL,
          error MEDIUMTEXT,
          products_data MEDIUMTEXT,

          PRIMARY KEY  (exec_id)
        ) $charset_collate;";
        
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        return dbDelta( $sql );

    }

    public function get_last_exec_values( array $cols, $status = null ) {

        $columns = implode( ',', $cols );
        $query = sprintf( "SELECT %s FROM $this->table ", $columns );
        if ( $status !== null ) {
            $query .= $this->wpdb->prepare( "WHERE status = %s ", $status );
        }
        $query .= "ORDER BY start_time DESC LIMIT 1";
        $result = $this->wpdb->get_row( $query, ARRAY_A );
        $output = [];
        if ( $result ) {
            foreach ( $cols as $col ) {
                $output[ $col ] = isset( $result[ $col ] ) ? $result[ $col ] : null;
            }
        }
        return $output;

    }
    

    public function clean_duplicate_dates() {

        return $this->wpdb->query(
            "DELETE t1
            FROM $this->table t1
            LEFT JOIN (
                SELECT MAX(start_time) as latest_time, DATE(start_time) as day
                FROM $this->table
                WHERE status = 'completed'
                GROUP BY day
            ) t2
            ON t1.start_time = t2.latest_time
            AND DATE(t1.start_time) = t2.day
            WHERE t2.latest_time IS NULL
            AND t1.status = 'completed'"
        );

    }

    public function is_full_exec() {

        if ( $this->get_active_exec() ) return true;
        return false;

    }

    public function get_active_exec_id() {

        return $this->wpdb->get_var( "SELECT exec_id FROM $this->table WHERE status = 'in-progress' ORDER BY start_time DESC LIMIT 1" );

    }

    public function get_active_exec() {

        return $this->wpdb->get_row( "SELECT * FROM $this->table WHERE status = 'in-progress' ORDER BY start_time DESC LIMIT 1" );

    }

    public function abort_active_exec( $exec_id = null ) {
        
        if ( ! $exec_id ) $exec_id = $this->exec_id;
        if ( ! $this->exec_id ) return null;
        update_option( 'zapbott_full_exec_heartbeat', '', true );
        return $this->update_exec( array( 'status' => 'aborted' ), $exec_id );

    }

    public function get_value( $column, $exec_id = null ) {

        if ( ! $exec_id ) $exec_id = $this->exec_id;
        $data = $this->wpdb->get_var( "SELECT $column FROM $this->table WHERE exec_id = '$exec_id'" );
        if ( !$data ) return null;
        if ( $column === 'products_data' ) return unserialize( $data );
        return $data;

    }

    public function get_execs_by_age( int $age, string $status = null ) {

        $query = "SELECT * FROM $this->table ";
        if ( $status !== null ) {
            $query .= "WHERE status = %s ";
        }
        $query .= "ORDER BY start_time DESC 
                    LIMIT %d";
        $args = [];
        if ( $status !== null ) {
            $args[] = sanitize_text_field( $status );
        }
        $args[] = $age;
        return array_reverse( $this->wpdb->get_results(
            $this->wpdb->prepare( $query, ...$args ),
            ARRAY_A
        ));

    }

    public function get_row_value( $row, string $column ) {

        if ( is_array( $row ) ) $row = (object) $row;
        if ( ! is_object( $row ) || ! $row ) return false;
        if ( ! $value = $row->$column ) return null;
        if ( $column === 'products_data' ) return unserialize( $value );
        return $value;

    }

    public function get_row_completed_products_ids( $row, $count = false ) {

        if ( ! is_object( $row ) ) return false;
        if ( ! $value = $row->products_data ) return null;
        $products_data = unserialize( $value );
        return $count === true ? count( array_keys( $products_data ) ) : array_keys( $products_data );

    }

    public function get_all_execs() {

        return $this->wpdb->get_results( "SELECT * FROM $this->table ORDER BY start_time DESC" );

    }

    public function get_last_completed_exec() {

        return $this->wpdb->get_row( "SELECT * FROM $this->table WHERE status = 'completed' ORDER BY start_time DESC LIMIT 1" );

    }

    public function get_execs_products_data( int $age, string $status = 'completed' ) {

        $query = "SELECT products_data FROM $this->table WHERE status = %s ORDER BY start_time DESC 
                    LIMIT %d";
        $args[] = sanitize_text_field( $status );
        $args[] = $age;
        $data = array_reverse( $this->wpdb->get_results(
            $this->wpdb->prepare( $query, ...$args ),
            ARRAY_A
        ));
        $products_data = [];
        foreach ( $data as $exec ) {
            $products_data[] = unserialize( $exec['products_data'] );
        }
        return $products_data;

    }

    public function get_completed_exec_by_date( $date ) {

        $formatted_date = date( 'Y-m-d', strtotime( $date ) );
        return $this->wpdb->get_row( $this->wpdb->prepare( 
            "SELECT * FROM $this->table WHERE status = 'completed' AND DATE(start_time) = %s ORDER BY start_time DESC LIMIT 1",
            $formatted_date
        ));

    }

    public function get_completed_product_ids( $exec_id = null, $count = false ) {

        if ( ! $exec_id ) $exec_id = $this->exec_id;
        $data = $this->wpdb->get_var( "SELECT products_data FROM $this->table WHERE exec_id = '$exec_id'" );
        if ( !$data ) return 0;
        $data = unserialize( $data );
        return $count === true ? count( array_keys( $data ) ) : array_keys( $data );

    }

    public function insert_exec( $data ) {

        if ( $this->is_full_exec() ) exit;
        if ( ! isset( $data['exec_id'] ) ) return null;
        $insert = $this->wpdb->insert( $this->table, $data );
        if ( $insert === false ) return false;
        $limit_db_rows = (int) get_option( 'zapbott_option_log_history_limit', 24 );
        if ( ! $limit_db_rows ) $limit_db_rows = 24;
        $this->limit_db_rows( $limit_db_rows );
        return $insert;

    }

    public function update_exec( $data, $exec_id = null ) {

        if ( ! $exec_id ) $exec_id = $this->exec_id;
        if ( ! $exec_id ) return false;
        return $this->wpdb->update( $this->table, $data, array( 'exec_id' => $exec_id ) );

    }

}

?>